import requests
from flask import Blueprint, jsonify, request, current_app
from config import Config

# 创建API蓝图
api_bp = Blueprint('api', __name__, url_prefix='/api')

@api_bp.route('/customers/<shop>')
def get_customers(shop):
    """
    获取客户列表
    """
    if shop not in Config.ACCESS_TOKENS:
        return jsonify({'error': 'Shop not authorized'}), 401
    
    token_data = Config.ACCESS_TOKENS[shop]
    access_token = token_data.get('access_token')
    
    if not access_token:
        return jsonify({'error': 'No access token available'}), 401
    
    try:
        # 调用Shoplazza API获取客户列表
        api_url = f"https://{shop}/openapi/2022-01/customers"
        headers = {
            'Access-Token': access_token,
            'Content-Type': 'application/json'
        }
        
        response = requests.get(api_url, headers=headers)
        response.raise_for_status()
        
        return jsonify({
            'shop': shop,
            'customers': response.json()
        })
        
    except Exception as e:
        current_app.logger.error(f"获取客户列表失败: {str(e)}")
        return jsonify({'error': 'Failed to fetch customers'}), 500

@api_bp.route('/products/<shop>')
def get_products(shop):
    """
    获取产品列表
    """
    if shop not in Config.ACCESS_TOKENS:
        return jsonify({'error': 'Shop not authorized'}), 401
    
    token_data = Config.ACCESS_TOKENS[shop]
    access_token = token_data.get('access_token')
    
    if not access_token:
        return jsonify({'error': 'No access token available'}), 401
    
    try:
        # 调用Shoplazza API获取产品列表
        api_url = f"https://{shop}/openapi/2020-01/products"
        headers = {
            'Access-Token': access_token,
            'Content-Type': 'application/json'
        }
        
        response = requests.get(api_url, headers=headers)
        response.raise_for_status()
        
        return jsonify({
            'shop': shop,
            'products': response.json()
        })
        
    except Exception as e:
        current_app.logger.error(f"获取产品列表失败: {str(e)}")
        return jsonify({'error': 'Failed to fetch products'}), 500

@api_bp.route('/orders/<shop>')
def get_orders(shop):
    """
    获取订单列表
    """
    if shop not in Config.ACCESS_TOKENS:
        return jsonify({'error': 'Shop not authorized'}), 401
    
    token_data = Config.ACCESS_TOKENS[shop]
    access_token = token_data.get('access_token')
    
    if not access_token:
        return jsonify({'error': 'No access token available'}), 401
    
    try:
        # 调用Shoplazza API获取订单列表
        api_url = f"https://{shop}/openapi/2020-01/orders"
        headers = {
            'Access-Token': access_token,
            'Content-Type': 'application/json'
        }
        
        response = requests.get(api_url, headers=headers)
        response.raise_for_status()
        
        return jsonify({
            'shop': shop,
            'orders': response.json()
        })
        
    except Exception as e:
        current_app.logger.error(f"获取订单列表失败: {str(e)}")
        return jsonify({'error': 'Failed to fetch orders'}), 500

@api_bp.route('/shop_info/<shop>')
def get_shop_info(shop):
    """
    获取商店信息
    """
    if shop not in Config.ACCESS_TOKENS:
        return jsonify({'error': 'Shop not authorized'}), 401
    
    token_data = Config.ACCESS_TOKENS[shop]
    access_token = token_data.get('access_token')
    
    if not access_token:
        return jsonify({'error': 'No access token available'}), 401
    
    try:
        # 调用Shoplazza API获取商店信息
        api_url = f"https://{shop}/openapi/2020-01/shop"
        headers = {
            'Access-Token': access_token,
            'Content-Type': 'application/json'
        }
        
        response = requests.get(api_url, headers=headers)
        response.raise_for_status()
        
        return jsonify({
            'shop': shop,
            'shop_info': response.json()
        })
        
    except Exception as e:
        current_app.logger.error(f"获取商店信息失败: {str(e)}")
        return jsonify({'error': 'Failed to fetch shop info'}), 500
