from flask import Flask, jsonify
from config import Config
from routes.auth import auth_bp
from routes.api import api_bp
from routes.webhook import webhook_bp
import logging

def create_app():
    """创建Flask应用"""
    app = Flask(__name__)
    
    # 配置应用
    app.config.from_object(Config)
    
    # 配置日志
    logging.basicConfig(level=logging.INFO)
    
    # 注册蓝图
    app.register_blueprint(auth_bp)
    app.register_blueprint(api_bp)
    app.register_blueprint(webhook_bp)
    
    # 根路由
    @app.route('/')
    def index():
        return jsonify({
            'message': 'Shoplazza OAuth2.0 Backend Service',
            'version': '1.0.0',
            'endpoints': {
                'auth': {
                    'install': '/auth/install?shop=your-shop.myshoplaza.com',
                    'callback': '/auth/shoplazza/callback',
                    'refresh_token': '/auth/refresh_token/<shop>',
                    'tokens': '/auth/tokens'
                },
                'api': {
                    'customers': '/api/customers/<shop>',
                    'products': '/api/products/<shop>',
                    'orders': '/api/orders/<shop>',
                    'shop_info': '/api/shop_info/<shop>'
                },
                'webhook': {
                    'shoplazza': '/webhook/shoplazza'
                }
            }
        })
    
    # 健康检查端点
    @app.route('/health')
    def health():
        return jsonify({
            'status': 'healthy',
            'authorized_shops': len(Config.ACCESS_TOKENS)
        })
    
    return app

if __name__ == '__main__':
    app = create_app()
    app.run(
        host='0.0.0.0',
        port=Config.PORT,
        debug=Config.DEBUG
    )
