#!/usr/bin/env python3
"""
修复Redis中的key格式问题
将错误的 i2i:* 格式迁移到正确的 item:similar:* 格式
"""
import redis
import sys
import os
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
from config.offline_config import REDIS_CONFIG

def fix_redis_keys():
    """修复Redis key格式"""

    # 连接Redis
    redis_client = redis.Redis(
        host=REDIS_CONFIG['host'],
        port=REDIS_CONFIG['port'],
        db=REDIS_CONFIG['db'],
        password=REDIS_CONFIG['password'],
        decode_responses=True
    )
    
    try:
        redis_client.ping()
        print("✓ Redis连接成功")
    except Exception as e:
        print(f"✗ Redis连接失败: {e}")
        return 1
    
    # 统计信息
    print("\n" + "="*80)
    print("当前Redis数据统计:")
    print("="*80)
    
    total_keys = redis_client.dbsize()
    print(f"总Key数量: {total_keys}")
    
    # 统计各类型key
    i2i_keys = redis_client.keys('i2i:*')
    item_similar_keys = redis_client.keys('item:similar:*')
    interest_keys = redis_client.keys('interest:*')
    
    print(f"错误格式 i2i:* 数量: {len(i2i_keys)}")
    print(f"正确格式 item:similar:* 数量: {len(item_similar_keys)}")
    print(f"interest:* 数量: {len(interest_keys)}")
    
    # 询问是否删除错误的key
    print("\n" + "="*80)
    print("⚠️  警告：即将删除所有 i2i:* 格式的key（错误格式）")
    print("="*80)
    
    if i2i_keys:
        print(f"\n将删除 {len(i2i_keys)} 个错误格式的key")
        print("示例:")
        for key in i2i_keys[:5]:
            print(f"  - {key}")
        
        response = input("\n确认删除？(yes/no): ")
        if response.lower() == 'yes':
            # 使用pipeline批量删除
            pipe = redis_client.pipeline()
            for key in i2i_keys:
                pipe.delete(key)
            pipe.execute()
            print(f"✓ 已删除 {len(i2i_keys)} 个错误格式的key")
        else:
            print("取消删除操作")
    else:
        print("✓ 没有需要删除的错误格式key")
    
    # 最终统计
    print("\n" + "="*80)
    print("清理后统计:")
    print("="*80)
    total_keys_after = redis_client.dbsize()
    print(f"总Key数量: {total_keys_after}")
    print(f"item:similar:* 数量: {len(redis_client.keys('item:similar:*'))}")
    print(f"interest:* 数量: {len(redis_client.keys('interest:*'))}")
    
    print("\n✓ 修复完成！现在可以使用修复后的加载脚本重新加载数据。")
    
    return 0


if __name__ == '__main__':
    sys.exit(fix_redis_keys())

