# README: 基于用户行为Session的Word2Vec模型训练与电子书推荐系统

## 目录

1. [项目概要](#项目概要)
2. [技术方案介绍](#技术方案介绍)
3. [代码细节说明](#代码细节说明)
4. [词向量与推荐系统的关联](#词向量与推荐系统的关联)

---

## 项目概要

本项目基于用户行为数据（Session）训练Word2Vec模型，用于电子书推荐系统。通过用户在一天内的阅读行为形成的Session数据，使用Word2Vec模型生成每本电子书的向量表示（embedding），并通过计算相似度来推荐书籍。

整个流程包括读取用户行为Session，训练Word2Vec模型，生成书籍的嵌入向量，并为每本书籍计算Top K相似书籍。模型学习的向量能够表达书籍之间的语义相似性，从而提高推荐系统的推荐准确性和个性化。

---

## 技术方案介绍

### 数据与Session定义

在电子书推荐场景中，用户一天内交互的书籍可以看作一个Session，即用户在一天内阅读或浏览的电子书形成一个序列。每个Session包含一组书籍ID，代表用户的行为轨迹。通过将这些Session输入Word2Vec模型，模型将学习到书籍之间的隐含关系，能够表示书籍的相似性。

### 词向量模型选择

当前实现中使用了`Word2Vec`模型的Skip-gram训练方式，该模型从上下文中预测词语（在此场景下即书籍）的关系。具体的实现基于`gensim`库，通过窗口大小（window）和嵌入维度（embedding size）等参数控制模型效果。

通过训练，模型将生成每本书籍的向量表示，并可以基于这些向量计算出每本书籍的Top K相似书籍。这种相似性可以应用到推荐系统中，用于向用户推荐相关或类似的电子书。

---

## 代码细节说明

下面我们对当前已有的训练代码及其主要部分进行说明。

### 1. **代码功能**

该代码通过加载包含用户Session（即用户一天内阅读的电子书ID）的输入文件，训练一个Word2Vec模型。随后，它保存了每个书籍ID对应的嵌入向量（embedding）及每本书籍的Top K相似书籍。

代码的主要功能包括：

- 读取用户行为数据，并将每个Session作为一行句子输入到Word2Vec模型中。
- 训练Word2Vec模型，并保存每个书籍ID的嵌入向量。
- 基于训练好的模型，计算每本书籍的Top K相似书籍，并输出结果。

### 2. **主要代码逻辑**

```python
def train_word2vec(input_file, output_dir, embedding_size=128, window=5, min_count=1, workers=4, top_k=200):
    """
    训练Word2Vec模型，并保存每个bid的embedding及top K相似的bid。
    
    :param input_file: 句子文件路径
    :param output_dir: 输出文件的目录路径
    :param embedding_size: 嵌入维度大小
    :param window: Word2Vec中的窗口大小
    :param min_count: Word2Vec中忽略频次低于min_count的词
    :param workers: 使用的线程数
    :param top_k: 每个bid的最相似bid的数量
    """
```

**输入参数解释：**

- `input_file`: 用户行为数据文件路径，文件内容为每行一个Session，Session是书籍ID的序列。
- `output_dir`: 输出目录路径，保存嵌入向量及相似度结果的目录。
- `embedding_size`: 词向量的维度大小，控制每本书籍ID最终的向量维度（默认为128）。
- `window`: Word2Vec中的窗口大小，决定了模型如何在上下文中学习书籍之间的关系。
- `min_count`: 最小词频，忽略在Session中出现频率低于该值的书籍。
- `workers`: 用于加速模型训练的并行线程数量。
- `top_k`: 每本书籍输出的Top K相似书籍数量。

**代码逻辑概述：**

1. **检查输出目录**：如果输出目录不存在，则创建该目录。
2. **加载数据**：从`input_file`中读取Session数据，每一行表示用户一天内阅读的书籍序列。
3. **训练Word2Vec模型**：使用`gensim`库中的`Word2Vec`方法，基于用户行为数据训练书籍的向量表示。
4. **保存嵌入向量**：将每个书籍ID的嵌入向量保存到指定文件中。
5. **计算Top K相似书籍**：对每本书籍，基于向量计算其最相似的Top K书籍，并将结果保存到文件中。

### 3. **核心代码片段说明**

- **训练Word2Vec模型**：
  ```python
  model = Word2Vec(sentences, vector_size=embedding_size, window=window, min_count=min_count, workers=workers)
  ```
  这行代码使用`Word2Vec`方法训练模型，基于用户行为数据生成书籍的嵌入向量。`vector_size`表示嵌入向量的维度，`window`表示窗口大小，`min_count`控制忽略频率低的书籍。

- **保存嵌入向量**：
  ```python
  with open(embedding_file, 'w') as f_out:
      for bid in model.wv.index_to_key:
          vector = model.wv[bid]
          f_out.write(f"{bid} {' '.join(map(str, vector))}\n")
  ```
  该片段将训练好的每本书籍的向量保存到文件中。每一行是一个书籍ID及其对应的向量。

- **计算Top K相似书籍**：
  ```python
  similar_bids = model.wv.most_similar(bid, topn=top_k)
  ```
  使用`most_similar`方法计算每本书籍的Top K相似书籍，这些书籍基于向量空间的相似度进行排序，最相似的书籍会靠近在推荐列表中。

---

## 词向量与推荐系统的关联

### 场景描述

在当前业务场景中，用户通过电子书平台阅读电子书，系统基于用户的阅读行为生成推荐列表。通过Word2Vec模型，系统能够学习书籍之间的隐含关系，并为用户推荐与其已阅读书籍相似的电子书。

### 推荐逻辑

利用Word2Vec模型生成的书籍向量，我们可以基于以下推荐逻辑为用户提供个性化的电子书推荐：

1. **相似书籍推荐**：当用户浏览或阅读了一本书后，系统可以根据该书的向量找到相似的书籍，并推荐给用户。
2. **用户行为扩展**：基于用户的一段阅读历史，系统可以将其历史中的书籍向量组合起来，找到与其行为最相似的其他书籍进行推荐。
3. **冷启动问题**：对于新书籍或新用户，可以通过书籍的内容标签或其他特征将其引入到向量空间，从而利用已有的模型进行推荐。

这种方式能够提高推荐的准确性和相关性，帮助用户发现更多符合其兴趣的书籍。
