#!/bin/bash

source /home/tw/miniconda3/etc/profile.d/conda.sh

# SearchEngine Setup and Startup Script
# This script sets up the environment and starts all services

set -e  # Exit on error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${GREEN}========================================${NC}"
echo -e "${GREEN}SearchEngine Setup Script${NC}"
echo -e "${GREEN}========================================${NC}"

# Change to project directory
cd "$(dirname "$0")"
PROJECT_ROOT=$(pwd)

echo -e "\n${YELLOW}Step 1: Setting up Conda environment${NC}"
# Check if conda is available
if ! command -v conda &> /dev/null; then
    echo -e "${RED}Error: conda not found. Please install Miniconda or Anaconda${NC}"
    exit 1
fi

# Source conda
source /home/tw/miniconda3/etc/profile.d/conda.sh

# Check if environment exists
if conda env list | grep -q "searchengine"; then
    echo -e "${GREEN}Environment 'searchengine' already exists${NC}"
    conda activate searchengine
else
    echo -e "${YELLOW}Creating conda environment 'searchengine'...${NC}"
    conda env create -f environment.yml
    conda activate searchengine
    echo -e "${GREEN}Environment created successfully!${NC}"
fi

# Verify environment
echo -e "\n${YELLOW}Current Python version:${NC}"
python --version

echo -e "\n${YELLOW}Step 2: Loading configuration${NC}"
# Check if .env exists
if [ ! -f ".env" ]; then
    echo -e "${YELLOW}Creating .env from .env.example...${NC}"
    cp .env.example .env
    echo -e "${GREEN}.env file created. Please update it with your actual configuration.${NC}"
fi

# Display configuration
echo -e "${GREEN}Configuration loaded:${NC}"
python -c "from config.env_config import print_config; print_config()"

echo -e "\n${YELLOW}Step 3: Checking Elasticsearch connection${NC}"
python -c "
from config.env_config import get_es_config
from utils.es_client import ESClient
es_config = get_es_config()
client = ESClient(hosts=[es_config['host']], username=es_config.get('username'), password=es_config.get('password'))
if client.ping():
    print('✓ Elasticsearch is reachable')
else:
    print('✗ Elasticsearch connection failed')
    exit(1)
"

echo -e "\n${GREEN}========================================${NC}"
echo -e "${GREEN}Setup Complete!${NC}"
echo -e "${GREEN}========================================${NC}"
echo ""
echo -e "Next steps:"
echo -e "  1. Ingest data: ${YELLOW}./scripts/ingest.sh${NC}"
echo -e "  2. Start backend: ${YELLOW}./scripts/start_backend.sh${NC}"
echo -e "  3. Start frontend: ${YELLOW}./scripts/start_frontend.sh${NC}"
echo ""
