#!/usr/bin/env python3
"""
测试脚本：模拟前端请求，检查后端返回的分面结果
"""

import sys
import json
import requests
import argparse
from pathlib import Path

sys.path.insert(0, str(Path(__file__).parent.parent))


def main():
    parser = argparse.ArgumentParser(description='测试分面API')
    parser.add_argument('--api-url', type=str, default='http://localhost:6002/search/', help='API URL')
    parser.add_argument('--tenant-id', type=str, required=True, help='Tenant ID')
    args = parser.parse_args()
    
    # 模拟前端的分面请求（与frontend/static/js/app.js一致）
    request_data = {
        "query": "",  # 空查询，获取所有数据
        "size": 10,
        "from": 0,
        "facets": [
            "category1_name",
            "specifications.color",
            "specifications.size",
            "specifications.material"
        ]
    }
    
    headers = {
        "Content-Type": "application/json",
        "X-Tenant-ID": args.tenant_id
    }
    
    try:
        print(f"发送请求到: {args.api_url}")
        print(f"Tenant ID: {args.tenant_id}")
        print(f"请求数据:")
        print(json.dumps(request_data, indent=2, ensure_ascii=False))
        print("\n" + "="*60)
        
        response = requests.post(args.api_url, json=request_data, headers=headers, timeout=30)
        
        if response.status_code != 200:
            print(f"API错误: {response.status_code}")
            print(response.text)
            return 1
        
        data = response.json()
        
        print("API响应:")
        print(f"  总结果数: {data.get('total', 0)}")
        print(f"  返回结果数: {len(data.get('results', []))}")
        
        facets = data.get('facets', [])
        print(f"\n分面数量: {len(facets)}")
        
        if not facets:
            print("\n⚠ 分面列表为空！")
            return 1
        
        print("\n" + "="*60)
        print("分面详情:")
        print("="*60)
        
        for i, facet in enumerate(facets, 1):
            print(f"\n{i}. {facet.get('field')}")
            print(f"   标签: {facet.get('label')}")
            print(f"   类型: {facet.get('type')}")
            print(f"   值数量: {len(facet.get('values', []))}")
            print(f"   总计数: {facet.get('total_count', 0)}")
            
            values = facet.get('values', [])
            if values:
                print(f"   前5个值:")
                for v in values[:5]:
                    print(f"     - {v.get('value')}: {v.get('count')}")
            else:
                print(f"   ⚠ 值列表为空！")
        
        # 检查specifications.color分面
        print("\n" + "="*60)
        print("检查specifications.color分面:")
        print("="*60)
        
        color_facet = None
        for facet in facets:
            if facet.get('field') == 'specifications.color':
                color_facet = facet
                break
        
        if color_facet:
            print("✓ 找到specifications.color分面")
            print(f"  值数量: {len(color_facet.get('values', []))}")
            if color_facet.get('values'):
                print("  前10个值:")
                for v in color_facet.get('values', [])[:10]:
                    print(f"    {v.get('value')}: {v.get('count')}")
            else:
                print("  ⚠ 值列表为空！")
        else:
            print("✗ 未找到specifications.color分面")
            print(f"  可用分面字段: {[f.get('field') for f in facets]}")
        
        # 输出完整JSON（便于调试）
        print("\n" + "="*60)
        print("完整分面JSON（前500字符）:")
        print("="*60)
        facets_json = json.dumps(facets, indent=2, ensure_ascii=False)
        print(facets_json[:500])
        
    except requests.exceptions.ConnectionError as e:
        print(f"\n连接错误: 无法连接到API服务器 {args.api_url}")
        print("请确保后端服务正在运行")
        return 1
    except Exception as e:
        print(f"\n错误: {e}")
        import traceback
        traceback.print_exc()
        return 1
    
    return 0


if __name__ == '__main__':
    sys.exit(main())

