#!/usr/bin/env python3
"""
翻译功能测试脚本。

测试内容：
1. 翻译提示词配置加载
2. 同步翻译（索引场景）
3. 异步翻译（查询场景）
4. 不同提示词的使用
5. 缓存功能
6. DeepL Context参数使用
"""

import sys
import os
from pathlib import Path

# Add parent directory to path
sys.path.insert(0, str(Path(__file__).parent.parent))

from config import ConfigLoader
from query.translator import Translator
import logging

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


def test_config_loading():
    """测试配置加载"""
    print("\n" + "="*60)
    print("测试1: 配置加载")
    print("="*60)
    
    try:
        config_loader = ConfigLoader()
        config = config_loader.load_config()
        
        print(f"✓ 配置加载成功")
        print(f"  翻译服务: {config.query_config.translation_service}")
        print(f"  翻译提示词配置:")
        for key, value in config.query_config.translation_prompts.items():
            print(f"    {key}: {value[:60]}..." if len(value) > 60 else f"    {key}: {value}")
        
        return config
    except Exception as e:
        print(f"✗ 配置加载失败: {e}")
        import traceback
        traceback.print_exc()
        return None


def test_translator_sync(config):
    """测试同步翻译（索引场景）"""
    print("\n" + "="*60)
    print("测试2: 同步翻译（索引场景）")
    print("="*60)
    
    if not config:
        print("✗ 跳过：配置未加载")
        return None
    
    try:
        translator = Translator(
            api_key=config.query_config.translation_api_key,
            use_cache=True,
            glossary_id=config.query_config.translation_glossary_id,
            translation_context=config.query_config.translation_context
        )
        
        # 测试商品标题翻译（使用product_title提示词）
        test_texts = [
            ("蓝牙耳机", "zh", "en", "product_title"),
            ("Wireless Headphones", "en", "zh", "product_title"),
        ]
        
        for text, source_lang, target_lang, prompt_type in test_texts:
            if prompt_type == "product_title":
                if target_lang == "zh":
                    prompt = config.query_config.translation_prompts.get('product_title_zh')
                else:
                    prompt = config.query_config.translation_prompts.get('product_title_en')
            else:
                if target_lang == "zh":
                    prompt = config.query_config.translation_prompts.get('default_zh')
                else:
                    prompt = config.query_config.translation_prompts.get('default_en')
            
            print(f"\n翻译测试:")
            print(f"  原文 ({source_lang}): {text}")
            print(f"  目标语言: {target_lang}")
            print(f"  提示词: {prompt[:50] if prompt else 'None'}...")
            
            result = translator.translate(
                text,
                target_lang=target_lang,
                source_lang=source_lang,
                prompt=prompt
            )
            
            if result:
                print(f"  结果: {result}")
                print(f"  ✓ 翻译成功")
            else:
                print(f"  ⚠ 翻译返回None（可能是mock模式或无API key）")
        
        return translator
        
    except Exception as e:
        print(f"✗ 同步翻译测试失败: {e}")
        import traceback
        traceback.print_exc()
        return None


def test_translator_async(config, translator):
    """测试异步翻译（查询场景）"""
    print("\n" + "="*60)
    print("测试3: 异步翻译（查询场景）")
    print("="*60)
    
    if not config or not translator:
        print("✗ 跳过：配置或翻译器未初始化")
        return
    
    try:
        query_text = "手机"
        target_langs = ['en']
        source_lang = 'zh'
        
        query_prompt = config.query_config.translation_prompts.get('query_zh')
        
        print(f"查询文本: {query_text}")
        print(f"目标语言: {target_langs}")
        print(f"提示词: {query_prompt}")
        
        # 异步模式（立即返回，后台翻译）
        results = translator.translate_multi(
            query_text,
            target_langs,
            source_lang=source_lang,
            context=config.query_config.translation_context,
            async_mode=True,
            prompt=query_prompt
        )
        
        print(f"\n异步翻译结果:")
        for lang, translation in results.items():
            if translation:
                print(f"  {lang}: {translation} (缓存命中)")
            else:
                print(f"  {lang}: None (后台翻译中...)")
        
        # 同步模式（等待完成）
        print(f"\n同步翻译（等待完成）:")
        results_sync = translator.translate_multi(
            query_text,
            target_langs,
            source_lang=source_lang,
            context=config.query_config.translation_context,
            async_mode=False,
            prompt=query_prompt
        )
        
        for lang, translation in results_sync.items():
            print(f"  {lang}: {translation}")
        
    except Exception as e:
        print(f"✗ 异步翻译测试失败: {e}")
        import traceback
        traceback.print_exc()


def test_cache():
    """测试缓存功能"""
    print("\n" + "="*60)
    print("测试4: 缓存功能")
    print("="*60)
    
    try:
        config_loader = ConfigLoader()
        config = config_loader.load_config()
        
        translator = Translator(
            api_key=config.query_config.translation_api_key,
            use_cache=True
        )
        
        test_text = "测试文本"
        target_lang = "en"
        source_lang = "zh"
        prompt = config.query_config.translation_prompts.get('default_zh')
        
        print(f"第一次翻译（应该调用API或返回mock）:")
        result1 = translator.translate(test_text, target_lang, source_lang, prompt=prompt)
        print(f"  结果: {result1}")
        
        print(f"\n第二次翻译（应该使用缓存）:")
        result2 = translator.translate(test_text, target_lang, source_lang, prompt=prompt)
        print(f"  结果: {result2}")
        
        if result1 == result2:
            print(f"  ✓ 缓存功能正常")
        else:
            print(f"  ⚠ 缓存可能有问题")
        
    except Exception as e:
        print(f"✗ 缓存测试失败: {e}")
        import traceback
        traceback.print_exc()


def test_context_parameter():
    """测试DeepL Context参数使用"""
    print("\n" + "="*60)
    print("测试5: DeepL Context参数")
    print("="*60)
    
    try:
        config_loader = ConfigLoader()
        config = config_loader.load_config()
        
        translator = Translator(
            api_key=config.query_config.translation_api_key,
            use_cache=False  # 禁用缓存以便测试
        )
        
        # 测试带context和不带context的翻译
        text = "手机"
        prompt = config.query_config.translation_prompts.get('query_zh')
        
        print(f"测试文本: {text}")
        print(f"提示词（作为context）: {prompt}")
        
        # 带context的翻译
        result_with_context = translator.translate(
            text,
            target_lang='en',
            source_lang='zh',
            prompt=prompt
        )
        print(f"\n带context翻译结果: {result_with_context}")
        
        # 不带context的翻译
        result_without_context = translator.translate(
            text,
            target_lang='en',
            source_lang='zh',
            prompt=None
        )
        print(f"不带context翻译结果: {result_without_context}")
        
        print(f"\n✓ Context参数测试完成")
        print(f"  注意：根据DeepL API，context参数影响翻译但不参与翻译本身")
        
    except Exception as e:
        print(f"✗ Context参数测试失败: {e}")
        import traceback
        traceback.print_exc()


def main():
    """主测试函数"""
    print("="*60)
    print("翻译功能测试")
    print("="*60)
    
    # 测试1: 配置加载
    config = test_config_loading()
    
    # 测试2: 同步翻译
    translator = test_translator_sync(config)
    
    # 测试3: 异步翻译
    test_translator_async(config, translator)
    
    # 测试4: 缓存功能
    test_cache()
    
    # 测试5: Context参数
    test_context_parameter()
    
    print("\n" + "="*60)
    print("测试完成")
    print("="*60)


if __name__ == '__main__':
    main()

