# 翻译功能测试说明

## 功能概述

本次更新实现了以下功能：

1. **翻译提示词配置**：支持中英文提示词，用于提高翻译质量
2. **DeepL Context参数**：提示词作为DeepL API的`context`参数传递（不参与翻译，仅提供上下文）
3. **同步/异步翻译**：
   - 索引场景：同步翻译，等待结果返回
   - 查询场景：异步翻译，立即返回缓存结果
4. **缓存机制**：翻译结果自动缓存，避免重复翻译

## 配置说明

### 配置文件位置

`config/config.yaml`

### 翻译提示词配置

```yaml
translation_prompts:
  # 商品标题翻译提示词
  product_title_zh: "请将原文翻译成中文商品SKU名称，要求：确保精确、完整地传达原文信息的基础上，语言简洁清晰、地道、专业。"
  product_title_en: "Translate the original text into an English product SKU name. Requirements: Ensure accurate and complete transmission of the original information, with concise, clear, authentic, and professional language."
  # query翻译提示词
  query_zh: "电商领域"
  query_en: "e-commerce domain"
  # 默认翻译用词
  default_zh: "电商领域"
  default_en: "e-commerce domain"
```

### 提示词使用规则

1. **商品标题翻译**：
   - 中文→英文：使用 `product_title_en`
   - 英文→中文：使用 `product_title_zh`

2. **其他字段翻译**（brief, description, vendor）：
   - 根据目标语言选择 `default_zh` 或 `default_en`

3. **查询翻译**：
   - 根据目标语言选择 `query_zh` 或 `query_en`

## 测试方法

### 1. 测试配置加载

```python
from config import ConfigLoader

config_loader = ConfigLoader()
config = config_loader.load_config()

# 检查翻译提示词配置
print(config.query_config.translation_prompts)
```

### 2. 测试同步翻译（索引场景）

```python
from query.translator import Translator
from config import ConfigLoader

config = ConfigLoader().load_config()
translator = Translator(
    api_key=config.query_config.translation_api_key,
    use_cache=True
)

# 测试商品标题翻译
text = "蓝牙耳机"
prompt = config.query_config.translation_prompts.get('product_title_en')
result = translator.translate(
    text,
    target_lang='en',
    source_lang='zh',
    prompt=prompt
)
print(f"翻译结果: {result}")
```

### 3. 测试异步翻译（查询场景）

```python
# 异步模式（立即返回，后台翻译）
results = translator.translate_multi(
    "手机",
    target_langs=['en'],
    source_lang='zh',
    async_mode=True,
    prompt=config.query_config.translation_prompts.get('query_zh')
)
print(f"异步结果: {results}")  # 可能包含None（后台翻译中）

# 同步模式（等待完成）
results_sync = translator.translate_multi(
    "手机",
    target_langs=['en'],
    source_lang='zh',
    async_mode=False,
    prompt=config.query_config.translation_prompts.get('query_zh')
)
print(f"同步结果: {results_sync}")
```

### 4. 测试文档转换器集成

```python
from indexer.document_transformer import SPUDocumentTransformer
import pandas as pd

# 创建模拟数据
spu_row = pd.Series({
    'id': 123,
    'tenant_id': '1',
    'title': '蓝牙耳机',
    'brief': '高品质无线蓝牙耳机',
    'description': '这是一款高品质的无线蓝牙耳机。',
    'vendor': '品牌A',
    # ... 其他字段
})

# 初始化转换器（带翻译器）
transformer = SPUDocumentTransformer(
    category_id_to_name={},
    searchable_option_dimensions=['option1', 'option2', 'option3'],
    tenant_config={'primary_language': 'zh', 'translate_to_en': True},
    translator=translator,
    translation_prompts=config.query_config.translation_prompts
)

# 转换文档
doc = transformer.transform_spu_to_doc(
    tenant_id='1',
    spu_row=spu_row,
    skus=pd.DataFrame(),
    options=pd.DataFrame()
)

print(f"title_zh: {doc.get('title_zh')}")
print(f"title_en: {doc.get('title_en')}")  # 应该包含翻译结果
```

### 5. 测试缓存功能

```python
# 第一次翻译（调用API）
result1 = translator.translate("测试文本", "en", "zh", prompt="电商领域")

# 第二次翻译（使用缓存）
result2 = translator.translate("测试文本", "en", "zh", prompt="电商领域")

assert result1 == result2  # 应该相同
```

## DeepL API Context参数说明

根据 [DeepL API文档](https://developers.deepl.com/api-reference/translate/request-translation)：

- `context` 参数：Additional context that can influence a translation but is not translated itself
- Context中的字符不计入计费
- Context用于提供翻译上下文，帮助提高翻译质量

我们的实现：
- 将提示词作为 `context` 参数传递给DeepL API
- Context不参与翻译，仅提供上下文信息
- 不同场景使用不同的提示词（商品标题、查询、默认）

## 运行完整测试

```bash
# 激活环境
source /home/tw/miniconda3/etc/profile.d/conda.sh
conda activate searchengine

# 运行测试脚本
python scripts/test_translation.py
```

## 验证要点

1. **配置加载**：确认所有提示词配置正确加载
2. **同步翻译**：索引时翻译结果正确填充到文档
3. **异步翻译**：查询时缓存命中立即返回，未命中后台翻译
4. **提示词使用**：不同场景使用正确的提示词
5. **缓存机制**：相同文本和提示词的翻译结果被缓存

## 注意事项

1. 需要配置 `DEEPL_AUTH_KEY` 环境变量或 `translation_api_key`
2. 如果没有API key，翻译器会返回原文（mock模式）
3. 缓存文件存储在 `.cache/translations.json`
4. Context参数中的字符不计入DeepL计费

