# API 快速参考 (v3.0)

## 基础搜索

```bash
POST /search/
{
  "query": "芭比娃娃",
  "size": 20
}
```

---

## 精确匹配过滤

```bash
{
  "filters": {
    "category_name": "手机",                    // 单值
    "category1_name": "服装",                  // 一级类目
    "vendor_zh.keyword": ["奇乐", "品牌A"],    // 多值（OR）
    "tags": "手机",                             // 标签
    // specifications 嵌套过滤
    "specifications": {
      "name": "color",
      "value": "white"
    }
  }
}
```

### Specifications 过滤

**单个规格**:
```bash
{
  "filters": {
    "specifications": {"name": "color", "value": "white"}
  }
}
```

**多个规格（按维度分组）**:
```bash
{
  "filters": {
    "specifications": [
      {"name": "color", "value": "white"},
      {"name": "size", "value": "256GB"}
    ]
  }
}
```
说明：不同维度（不同name）是AND关系，相同维度（相同name）的多个值是OR关系。

---

## 范围过滤

```bash
{
  "range_filters": {
    "min_price": {
      "gte": 50,    // >=
      "lte": 200    // <=
    }
  }
}
```

**操作符**: `gte` (>=), `gt` (>), `lte` (<=), `lt` (<)

---

## 分面搜索

### 简单模式

```bash
{
  "facets": ["category1_name", "category2_name", "category3_name", "specifications"]
}
```

### Specifications 分面

**所有规格名称**:
```bash
{
  "facets": ["specifications"]  // 返回所有name及其value列表
}
```

**指定规格名称**:
```bash
{
  "facets": ["specifications.color", "specifications.size", "specifications.material"]  // 只返回指定name的value列表
}
```

### 高级模式

```bash
{
  "facets": [
    {"field": "category1_name", "size": 15},
    {
      "field": "min_price",
      "type": "range",
      "ranges": [
        {"key": "0-50", "to": 50},
        {"key": "50-100", "from": 50, "to": 100}
      ]
    },
    "specifications",              // 所有规格名称
    "specifications.color",         // 指定规格名称
    "specifications.size",
    "specifications.material"
  ]
}
```

---

## SKU筛选维度

**功能**: 按指定维度对每个SPU下的SKU进行分组，每组只返回第一个SKU。

```bash
{
  "query": "芭比娃娃",
  "sku_filter_dimension": "color"    // 按颜色筛选（假设option1_name="color"）
}
```

**支持的维度值**:
- `option1`, `option2`, `option3`: 直接使用选项字段
- 规格名称（如 `color`, `size`）: 通过 `option1_name`、`option2_name`、`option3_name` 匹配

**示例**:
```bash
// 按选项1筛选
{"sku_filter_dimension": "option1"}

// 按颜色筛选（如果option1_name="color"）
{"sku_filter_dimension": "color"}

// 按尺寸筛选（如果option2_name="size"）
{"sku_filter_dimension": "size"}
```

**性能说明**: 在应用层过滤，不影响ES查询性能，只对返回结果进行过滤。

---

## 排序

```bash
{
  "sort_by": "min_price",
  "sort_order": "asc"    // asc 或 desc
}
```

---

## 布尔表达式

```bash
{
  "query": "玩具 AND (乐高 OR 芭比) ANDNOT 电动"
}
```

**操作符优先级**: `()` > `ANDNOT` > `AND` > `OR` > `RANK`

---

## 分页

```bash
{
  "size": 20,      // 每页数量
  "from": 0        // 偏移量（第1页=0，第2页=20）
}
```

---

## 完整示例

```bash
POST /search/
Headers: X-Tenant-ID: 162
{
  "query": "手机",
  "size": 20,
  "from": 0,
  "language": "zh",
  "filters": {
    "category_name": "手机",
    "category1_name": "电子产品",
    "specifications": {"name": "color", "value": "white"}
  },
  "range_filters": {
    "min_price": {"gte": 50, "lte": 200}
  },
  "facets": [
    {"field": "category1_name", "size": 15},
    {"field": "category2_name", "size": 15},
    "specifications.color",
    "specifications.size"
  ],
  "sort_by": "min_price",
  "sort_order": "asc",
  "sku_filter_dimension": "color"    // 可选：按颜色筛选SKU
}
```

---

## 响应格式

```json
{
  "results": [
    {
      "spu_id": "12345",
      "title": "商品标题",
      "brief": "短描述",
      "description": "详细描述",
      "vendor": "供应商",
      "category": "类目",
      "category_path": "类目/路径",
      "category_name": "类目名称",
      "category1_name": "一级类目",
      "category2_name": "二级类目",
      "category3_name": "三级类目",
      "tags": ["标签1", "标签2"],
      "price": 99.99,
      "compare_at_price": 149.99,
      "sku_prices": [99.99, 109.99],
      "total_inventory": 500,
      "specifications": [
        {"sku_id": "sku_001", "name": "color", "value": "white"}
      ],
      "skus": [...],  // 如果指定了sku_filter_dimension，则返回过滤后的SKU（每个维度值一个）
      "relevance_score": 8.5
    }
  ],
  "total": 118,
  "max_score": 8.5,
  "took_ms": 45,
  "facets": [
    {
      "field": "category1_name",
      "label": "category1_name",
      "type": "terms",
      "values": [
        {
          "value": "手机",
          "label": "手机",
          "count": 85,
          "selected": false
        }
      ]
    },
    {
      "field": "specifications.color",
      "label": "color",
      "type": "terms",
      "values": [
        {
          "value": "white",
          "label": "white",
          "count": 30,
          "selected": false
        }
      ]
    }
  ]
}
```

---

## 其他端点

```bash
POST /search/image
{
  "image_url": "https://example.com/image.jpg",
  "size": 20
}

GET /search/suggestions?q=芭&size=5

GET /search/instant?q=玩具&size=5

GET /search/{doc_id}

GET /admin/health
GET /admin/config
GET /admin/stats
```

---

## Python 快速示例

```python
import requests

result = requests.post(
    'http://localhost:6002/search/',
    headers={'X-Tenant-ID': '2'},
    json={
        "query": "手机",
        "language": "zh",
        "filters": {"category_name": "手机"},
        "range_filters": {"min_price": {"gte": 50, "lte": 200}},
        "facets": ["category1_name", "specifications"],
        "sort_by": "min_price",
        "sort_order": "asc"
    }
).json()

print(f"找到 {result['total']} 个结果")
```

---

## JavaScript 快速示例

```javascript
const result = await fetch('http://localhost:6002/search/', {
    method: 'POST',
    headers: {
        'Content-Type': 'application/json',
        'X-Tenant-ID': '2'
    },
    body: JSON.stringify({
        query: "手机",
        language: "zh",
        filters: {category_name: "手机"},
        range_filters: {min_price: {gte: 50, lte: 200}},
        facets: ["category1_name", "specifications"],
        sort_by: "min_price",
        sort_order: "asc"
    })
}).then(r => r.json());

console.log(`找到 ${result.total} 个结果`);
```

---
