# Amazon商品数据爬虫使用说明

## 概述

这是一个使用万邦API爬取亚马逊商品数据的Python脚本。脚本会读取查询列表文件，逐个请求API，并将结果保存为JSON文件。

## 文件说明

- `amazon_crawler.py` - 基础版爬虫脚本
- `amazon_crawler_v2.py` - 增强版爬虫脚本（推荐使用）
- `config.example.py` - 配置文件示例
- `queries.txt` - 搜索关键词列表（每行一个）
- `amazon_results/` - 结果保存目录（自动创建）
- `amazon_crawler.log` - 运行日志文件

## 快速开始

### 1. 安装依赖

```bash
pip install requests
```

### 2. 配置API密钥

有三种方式配置API密钥：

#### 方式1：使用配置文件（推荐）

```bash
cd data_crawling
cp config.example.py config.py
# 编辑 config.py，填入真实的API密钥
```

#### 方式2：使用命令行参数

```bash
python amazon_crawler_v2.py --key YOUR_KEY --secret YOUR_SECRET
```

#### 方式3：使用环境变量

```bash
export ONEBOUND_API_KEY="your_key_here"
export ONEBOUND_API_SECRET="your_secret_here"
python amazon_crawler_v2.py
```

### 3. 运行爬虫

#### 基础版运行

```bash
# 编辑 amazon_crawler.py，填入API密钥
python amazon_crawler.py
```

#### 增强版运行

```bash
# 使用默认配置
python amazon_crawler_v2.py

# 自定义参数
python amazon_crawler_v2.py \
  --key YOUR_KEY \
  --secret YOUR_SECRET \
  --queries queries.txt \
  --delay 2.0 \
  --start 0 \
  --max 100 \
  --output amazon_results
```

## 命令行参数说明

| 参数 | 说明 | 默认值 |
|------|------|--------|
| `--key` | API Key | 从配置文件或环境变量读取 |
| `--secret` | API Secret | 从配置文件或环境变量读取 |
| `--queries` | 查询文件路径 | `data_crawling/queries.txt` |
| `--delay` | 请求间隔（秒） | 2.0 |
| `--start` | 起始索引（断点续爬） | 0 |
| `--max` | 最大爬取数量 | None（全部） |
| `--output` | 结果保存目录 | `data_crawling/amazon_results` |

## 使用示例

### 示例1：测试前10个查询

```bash
python amazon_crawler_v2.py --max 10
```

### 示例2：从第100个查询开始继续爬取

```bash
python amazon_crawler_v2.py --start 100
```

### 示例3：加快爬取速度（减少延迟）

```bash
python amazon_crawler_v2.py --delay 1.0
```

### 示例4：使用自定义查询文件

```bash
python amazon_crawler_v2.py --queries my_queries.txt
```

### 示例5：完整参数示例

```bash
python amazon_crawler_v2.py \
  --key "your_api_key" \
  --secret "your_api_secret" \
  --queries queries.txt \
  --delay 2.0 \
  --start 0 \
  --max 100 \
  --output my_results
```

## API参数说明

根据万邦API文档，支持以下搜索参数：

| 参数 | 说明 | 示例 |
|------|------|------|
| `q` | 搜索关键字 | "Mobile Phone Holder" |
| `start_price` | 开始价格 | 10 |
| `end_price` | 结束价格 | 100 |
| `page` | 页码 | 1 |
| `cat` | 分类ID | - |
| `discount_only` | 仅折扣商品 | yes/no |
| `sort` | 排序方式 | - |
| `page_size` | 每页数量 | 100 |
| `cache` | 使用缓存 | yes/no |
| `result_type` | 返回格式 | json |
| `lang` | 语言 | cn/en |

## 结果文件格式

每个查询的结果保存为单独的JSON文件，文件名格式：

```
0001_Bohemian_Maxi_Dress.json
0002_Vintage_Denim_Jacket.json
...
```

JSON文件结构示例：

```json
{
  "items": {
    "item": [
      {
        "detail_url": "https://www.amazon.com/...",
        "num_iid": "B07F8S18D5",
        "pic_url": "https://...",
        "price": "9.99",
        "reviews": "53812",
        "sales": 10000,
        "stars": "4.7",
        "title": "商品标题"
      }
    ],
    "page": "1",
    "page_size": 100,
    "real_total_results": 700,
    "q": "搜索词"
  },
  "error_code": "0000",
  "reason": "ok"
}
```

## 日志文件

运行日志保存在 `amazon_crawler.log`，包含：

- 每个请求的状态
- 成功/失败统计
- 错误信息
- 爬取进度

示例日志：

```
2025-01-07 10:00:00 - INFO - Amazon爬虫启动
2025-01-07 10:00:01 - INFO - [1/5024] (0.0%) - Bohemian Maxi Dress
2025-01-07 10:00:02 - INFO - ✓ 成功: Bohemian Maxi Dress - 获得 700 个结果
```

## 注意事项

### 1. API限制

- 注意API的请求限制（每日/每月配额）
- 建议设置适当的延迟时间（--delay参数）
- 从API响应中可以查看配额信息：`api_info` 字段

### 2. 断点续爬

如果爬取中断，可以使用 `--start` 参数从指定索引继续：

```bash
# 从第1000个查询继续
python amazon_crawler_v2.py --start 1000
```

### 3. 查询文件格式

`queries.txt` 文件要求：
- UTF-8 编码
- 每行一个查询关键词
- 空行会被自动跳过

### 4. 错误处理

- 请求失败的查询会保存错误信息到JSON文件
- 所有错误都会记录在日志文件中
- 可以通过日志文件查找失败的查询并重新爬取

### 5. 结果分析

可以编写脚本统计和分析爬取结果：

```python
import json
from pathlib import Path

results_dir = Path("amazon_results")
total_items = 0

for json_file in results_dir.glob("*.json"):
    with open(json_file) as f:
        data = json.load(f)
        if data.get('error_code') == '0000':
            items = data.get('items', {}).get('item', [])
            total_items += len(items)

print(f"共爬取 {total_items} 个商品")
```

## 性能优化

### 1. 并发爬取

当前脚本是串行爬取，如果需要提高速度，可以：
- 将查询列表分割成多个文件
- 使用多个进程同时运行
- 或修改脚本支持多线程/异步请求

### 2. 内存优化

如果查询数量很大（如5024个），建议：
- 使用 `--max` 参数分批爬取
- 定期清理日志文件

### 3. 网络优化

- 选择网络稳定的环境运行
- 可以增加重试机制
- 使用代理（如有需要）

## 故障排除

### 问题1：API密钥错误

```
错误: 未配置API密钥！
```

**解决**：检查配置文件、命令行参数或环境变量是否正确设置

### 问题2：请求超时

```
请求失败: timeout
```

**解决**：
- 检查网络连接
- 增加超时时间（修改脚本中的 `timeout` 参数）
- 检查API服务是否正常

### 问题3：API配额耗尽

```
API返回错误: quota exceeded
```

**解决**：
- 等待配额重置
- 升级API套餐
- 使用 `--max` 参数限制爬取数量

### 问题4：文件编码错误

```
UnicodeDecodeError
```

**解决**：确保 `queries.txt` 是 UTF-8 编码

## 扩展功能

如果需要扩展功能，可以修改脚本添加：

1. **多页爬取**：每个查询爬取多页结果
2. **数据过滤**：按价格、评分等条件过滤商品
3. **数据库存储**：将结果存入数据库而非JSON文件
4. **重试机制**：失败的请求自动重试
5. **代理支持**：通过代理服务器发送请求
6. **异步请求**：使用 aiohttp 实现并发爬取

## 联系方式

如有问题，请查看：
- 万邦API文档：`万邦API_亚马逊.md`
- 项目README：`README.md`
- 日志文件：`amazon_crawler.log`

