"""
Configuration utility functions.

Helper functions for working with SearchConfig objects.
"""

from typing import Dict, List
from .config_loader import SearchConfig


def get_match_fields_for_index(config: SearchConfig, index_name: str = "default") -> List[str]:
    """
    Generate match fields list with boost from field_boosts.
    
    Args:
        config: SearchConfig instance
        index_name: Name of the index domain (default: "default")
    
    Returns:
        List of field names with boost, e.g., ["title_zh^3.0", "brief_zh^1.5"]
    """
    # Find the index config
    index_config = None
    for idx in config.indexes:
        if idx.name == index_name:
            index_config = idx
            break
    
    if not index_config:
        return []
    
    # Generate match fields with boost
    match_fields = []
    for field_name in index_config.fields:
        # Get field boost from field_boosts dictionary
        field_boost = config.field_boosts.get(field_name, 1.0)
        
        # Combine index boost and field boost
        total_boost = index_config.boost * field_boost
        
        if total_boost != 1.0:
            match_fields.append(f"{field_name}^{total_boost}")
        else:
            match_fields.append(field_name)
    
    return match_fields


def get_domain_fields(config: SearchConfig) -> Dict[str, List[str]]:
    """
    Generate domain-specific match fields from all index configs.
    
    Args:
        config: SearchConfig instance
    
    Returns:
        Dictionary mapping domain name to list of match fields
    """
    domain_fields = {}
    for index_config in config.indexes:
        domain_fields[index_config.name] = get_match_fields_for_index(config, index_config.name)
    return domain_fields
