"""
Centralized configuration management for SearchEngine.

Loads configuration from environment variables and .env file.
This module provides a single point for loading .env and setting defaults.
All configuration variables are exported directly - no need for getter functions.
"""

import os
from pathlib import Path
from dotenv import load_dotenv

# Load .env file from project root
PROJECT_ROOT = Path(__file__).parent.parent
load_dotenv(PROJECT_ROOT / '.env')


# Elasticsearch Configuration
ES_CONFIG = {
    'host': os.getenv('ES_HOST', 'http://localhost:9200'),
    'username': os.getenv('ES_USERNAME'),
    'password': os.getenv('ES_PASSWORD'),
}

# Redis Configuration
REDIS_CONFIG = {
    'host': os.getenv('REDIS_HOST', 'localhost'),
    'port': int(os.getenv('REDIS_PORT', 6479)),
    'password': os.getenv('REDIS_PASSWORD'),
}

# DeepL API Key
DEEPL_AUTH_KEY = os.getenv('DEEPL_AUTH_KEY')

# API Service Configuration
API_HOST = os.getenv('API_HOST', '0.0.0.0')
API_PORT = int(os.getenv('API_PORT', 6002))
# API_BASE_URL: 如果未设置，根据API_HOST构建（0.0.0.0使用localhost）
API_BASE_URL = os.getenv('API_BASE_URL')
if not API_BASE_URL:
    API_BASE_URL = f'http://localhost:{API_PORT}' if API_HOST == '0.0.0.0' else f'http://{API_HOST}:{API_PORT}'

# Model Directories
TEXT_MODEL_DIR = os.getenv('TEXT_MODEL_DIR', '/data/tw/models/bge-m3')
IMAGE_MODEL_DIR = os.getenv('IMAGE_MODEL_DIR', '/data/tw/models/cn-clip')

# Cache Directory
CACHE_DIR = os.getenv('CACHE_DIR', '.cache')

# MySQL Database Configuration (Shoplazza)
DB_CONFIG = {
    'host': os.getenv('DB_HOST'),
    'port': int(os.getenv('DB_PORT', 3306)) if os.getenv('DB_PORT') else 3306,
    'database': os.getenv('DB_DATABASE'),
    'username': os.getenv('DB_USERNAME'),
    'password': os.getenv('DB_PASSWORD'),
}


def print_config():
    """Print current configuration (with sensitive data masked)."""
    print("=" * 60)
    print("SearchEngine Configuration")
    print("=" * 60)

    print("\nElasticsearch:")
    print(f"  Host: {ES_CONFIG['host']}")
    print(f"  Username: {ES_CONFIG['username']}")
    print(f"  Password: {'*' * 10 if ES_CONFIG['password'] else 'None'}")

    print("\nRedis:")
    print(f"  Host: {REDIS_CONFIG['host']}")
    print(f"  Port: {REDIS_CONFIG['port']}")
    print(f"  Password: {'*' * 10 if REDIS_CONFIG['password'] else 'None'}")

    print("\nDeepL:")
    print(f"  API Key: {'*' * 10 if DEEPL_AUTH_KEY else 'None (translation disabled)'}")

    print("\nAPI Service:")
    print(f"  Host: {API_HOST}")
    print(f"  Port: {API_PORT}")

    print("\nModels:")
    print(f"  Text Model: {TEXT_MODEL_DIR}")
    print(f"  Image Model: {IMAGE_MODEL_DIR}")

    print("\nCache:")
    print(f"  Cache Directory: {CACHE_DIR}")

    print("\nMySQL Database:")
    print(f"  Host: {DB_CONFIG['host']}")
    print(f"  Port: {DB_CONFIG['port']}")
    print(f"  Database: {DB_CONFIG['database']}")
    print(f"  Username: {DB_CONFIG['username']}")
    print(f"  Password: {'*' * 10 if DB_CONFIG['password'] else 'None'}")

    print("=" * 60)


if __name__ == "__main__":
    print_config()
