"""
Admin API routes for configuration and management.
"""

from fastapi import APIRouter, HTTPException
from typing import Dict

from ..models import HealthResponse, ErrorResponse

router = APIRouter(prefix="/admin", tags=["admin"])


@router.get("/health", response_model=HealthResponse)
async def health_check():
    """
    Health check endpoint.

    Returns service status and Elasticsearch connectivity.
    """
    try:
        from ..app import get_es_client, get_config

        es_client = get_es_client()
        config = get_config()

        # Check ES connectivity
        es_status = "connected" if es_client.ping() else "disconnected"

        return HealthResponse(
            status="healthy" if es_status == "connected" else "unhealthy",
            elasticsearch=es_status
        )

    except Exception as e:
        return HealthResponse(
            status="unhealthy",
            elasticsearch="error"
        )


@router.get("/config")
async def get_configuration():
    """
    Get current search configuration (sanitized).
    """
    try:
        from ..app import get_config

        config = get_config()

        return {
            "es_index_name": config.es_index_name,
            "num_field_boosts": len(config.field_boosts),
            "num_indexes": len(config.indexes),
            "supported_languages": config.query_config.supported_languages,
            "ranking_expression": config.ranking.expression,
            "spu_enabled": config.spu_config.enabled
        }

    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/rewrite-rules")
async def update_rewrite_rules(rules: Dict[str, str]):
    """
    Update query rewrite rules.

    Args:
        rules: Dictionary of pattern -> replacement mappings
    """
    try:
        from ..app import get_query_parser

        query_parser = get_query_parser()
        query_parser.update_rewrite_rules(rules)

        return {
            "status": "success",
            "message": f"Updated {len(rules)} rewrite rules"
        }

    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/rewrite-rules")
async def get_rewrite_rules():
    """
    Get current query rewrite rules.
    """
    try:
        from ..app import get_query_parser

        query_parser = get_query_parser()
        rules = query_parser.get_rewrite_rules()

        return {
            "rules": rules,
            "count": len(rules)
        }

    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/stats")
async def get_index_stats():
    """
    Get index statistics.
    """
    try:
        from ..app import get_es_client, get_config

        es_client = get_es_client()
        config = get_config()

        # Get document count
        doc_count = es_client.count(config.es_index_name)

        # Get index size (if available)
        try:
            stats = es_client.client.indices.stats(index=config.es_index_name)
            size_in_bytes = stats['indices'][config.es_index_name]['total']['store']['size_in_bytes']
            size_mb = size_in_bytes / (1024 * 1024)
        except:
            size_mb = None

        return {
            "index_name": config.es_index_name,
            "document_count": doc_count,
            "size_mb": round(size_mb, 2) if size_mb else None
        }

    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))
